#! /usr/bin/env bash

# INSTRUCTIONS
#
# Query the loopvid API for all responses and save it to a file, e.g.
#
# curl --fail https://loopvid.kastden.org/blog/json/ -o videos.json
#
# Then run this script from within the folder where you want the videos to be
# downloaded to
#
# ./download_loopvid.sh videos.json


# REQUIREMENTS
# jq           https://stedolan.github.io/jq/
# curl         https://curl.haxx.se/
# mkvtoolnix   https://mkvtoolnix.download/
# exiftool     https://exiftool.org/


if [ -z "$1" ]; then echo "Usage: $0 videos.json" >&2; exit 1; fi

# output a line for each video in the form "<name>;<tag1>-<tag2>-...;<source>"
filter='.entries[]|(.name+";"+(.tags|join("-"))+";"+.source_text+"  "+.source_url)'
videos=(); while read line; do videos+=("$line"); done < <(jq -r "$filter" "$1")

# iterate through the videos, downloading whichever ones are not present in the
# current working directory
i=0
for line in "${videos[@]}"; do
    # print progress
    echo -en "\r$((++i))/${#videos[@]}"

    # parse line
    name="${line%%;*}"
    rest="${line#*;}"

    # if the name contains multiple formats, keep the first and drop the rest
    name="${name%,*}"

    tags="${rest%%;*}"
    source_str="${rest#*;}"
    # drop trailing whitespace if the source url is missing
    source_str="${source_str%  }"

    if [ -z "$name" -o -z "$tags" -o -z "$source_str" ]; then
        echo "Malformed line: $line" >&2; exit 2
    fi

    # generate output filenames
    base_name="${name/\//_}-${tags}"
    webm="${base_name}.webm"
    mp4="${base_name}.mp4"

    # check if the video has already been downloaded
    if [ -e "$webm" -o -e "$mp4" ]; then
        continue
    fi

    # attempt to download video as webm, and as mp4 if that fails
    base_url="https://kastden.org/_loopvid_media/${name}"
    webm_url="${base_url}.webm"
    mp4_url="${base_url}.mp4"

    echo -e "\nAttempting to download $name\n"

    # exit if we couldn't download either format, or if the user interrupted
    # the execution
    curl --fail --retry 20 --remote-time -o "$webm" "$webm_url" || \
        curl --fail --retry 20 --remote-time -o "$mp4" "$mp4_url" || exit 3

    # tag the file we just downloaded
    echo -e "\nAttempting to add source as metadata\n"

    if [ -e "$webm" ]; then
        # manually preserve the file's modification time, since mkvpropedit
        # doesn't support it
        mod_time="$(stat -c "%y" "$webm")"

        mkvpropedit -s "title=$source_str" "$webm" || exit 4

        touch -d "$mod_time" "$webm" || exit 5
    elif [ -e "$mp4" ]; then
        exiftool -P -overwrite_original -Title="$source_str" "$mp4" || exit 4
    fi

    echo -e "\n$name was successfully downloaded and tagged\n"
done

echo -e "\nAll done"
