#! /usr/bin/env python3
'''Functions for validating and normalizing user input'''

import re
from datetime import datetime

from .errors import DBError


# list of all hosts with videos present in the db
db_hosts = [
    'gc', 'kd', 'ko', 'ky', 'lv', 'm2', 'mf', 'mm', 'ni', 'nn', 'pc', 'pf',
    'pi',
]

_valid_entry_keys = [
    'posted_at', 'source_text', 'source_url', 'urls', 'name', 'tags',
    'has_audio', 'watch_url', 'views',
]

# default parameters accepted by pages
_accepted_vars = [
    'order', 'since', 'until', 'offset', 'limit', 'type', 'names', 'tags',
    'source', 'hosts', 'keys',
]

_list_separator = ','
_names_list_separators = [';', '+', ' ']
# the preferred separator
names_list_separator = '+'

_video_basename_regex = re.compile('^[^/]{2}/[^,&]+')
_valid_tag_regex = re.compile('^[0-9a-z_-]+$')


def split_names_list(list_string):
    '''Split a list of entry names.'''
    if not list_string:
        return []

    # try to split with all possible separators
    names = None
    for separator in _names_list_separators:
        if separator in list_string:
            names = list_string.split(separator)
            break
    # no separators present, return list with a single item
    if not names:
        names = [list_string]
    return names


def validate_tags(tags, fail=False):
    '''Returns list of fixed tags, without including malformed ones. If
    fail=True, raise error on malformed tags.'''
    good_tags = []
    for tag in tags:
        tag = tag.strip().lower().replace(' ', '_')
        if _valid_tag_regex.match(tag):
            good_tags.append(tag)
        elif fail:
            raise DBError('invalid tag name: {}'.format(tag))
    return good_tags


def validate_params(user_params, strict=True, defaults=None,
                    accepted_vars=None):
    '''Turn information extracted from the path and parameters passed by the
    user into parameters for querying the DB.'''
    # pylint: disable = not-an-iterable

    # accept all known keys by default
    if not accepted_vars:
        accepted_vars = _accepted_vars

    # defaults
    params = {
        # sort from newest to oldest
        'order': 'descending',
        # sort by post time
        'sort_by': 'posted_at',
        # accept all posted_at times
        'since': None,
        'until': None,
        'return_type': 'list',
        # start at the first entry
        'offset': None,
        # don't limit output to a number of entries
        'limit': None,
        # don't filter by tags
        'tags': None,
        # join tags with OR when multiple tags are present
        'tags_op': 'or',
        # don't filter by entry name
        'names': None,
        # don't filter by source
        'source': None,
        # don't filter by host
        'hosts': None,
        # don't filter keys
        'keys': None,
        # expand entries
        'expand': True,
        # entries will not be filtered in any way
        '_filtered' : False,
    }
    if defaults:
        for key in defaults:
            if key in params:
                params[key] = defaults[key]

    # check all variables
    if strict:
        for var in user_params:
            if not var in _accepted_vars:
                raise ValueError(u'Invalid argument: {}; expected {}'.format(
                    var, _accepted_vars))

    # order
    if 'order' in accepted_vars and 'order' in user_params:
        order = user_params['order']
        if not order in ['asc', 'ascending', 'desc', 'descending']:
            # raise error or ignore
            if strict:
                raise ValueError(
                    f'Invalid odering: "{order}". Expected "ascending" '
                    f'or "descending"'
                )
        elif order.startswith('asc'):
            params['order'] = 'ascending'

    if 'sort_by' in accepted_vars and 'sort_by' in user_params:
        sort_by = user_params['sort_by']
        if not sort_by in ['posted_at', 'views', 'random']:
            # raise error or ignore
            if strict:
                raise ValueError(f'Invalid sort by key: "{sort_by}". '
                                 f'Expected posted_at, views or random')
        else:
            params['sort_by'] = sort_by

    # since, until
    if 'since' in accepted_vars and 'since' in user_params:
        since = user_params['since']
        if since:
            if strict:
                # convert to datetime then back so strptime can raise ValueError if
                # necessary
                since = datetime.strptime(since, '%Y-%m-%d %H:%M:%S.%f')
                params['since'] = since.strftime('%Y-%m-%d %H:%M:%S.%f')
            else:
                params['since'] = since

    if 'until' in accepted_vars and 'until' in user_params:
        until = user_params['until']
        if until:
            if strict:
                # convert to datetime then back so strptime can raise ValueError if
                # necessary
                until = datetime.strptime(until, '%Y-%m-%d %H:%M:%S.%f')
                params['until'] = until.strftime('%Y-%m-%d %H:%M:%S.%f')
            else:
                params['until'] = until

    # number of entries, offset
    for var in ['limit', 'offset']:
        if var in accepted_vars and var in user_params:
            try:
                params[var] = int(user_params[var])
            except ValueError:
                if strict:
                    raise

    # return type
    if 'type' in accepted_vars and 'type' in user_params:
        return_type = user_params['type']
        if not return_type in ['list', 'dict']:
            # raise error or ignore
            if strict:
                raise ValueError(f'Invalid return type: "{return_type}". '
                                 f'Expected "list" or "dict"')
        elif return_type == 'dict':
            params['return_type'] = 'dict'

    # names
    if 'names' in accepted_vars and 'names' in user_params and user_params['names']:
        params['names'] = split_names_list(user_params['names'])

    # source
    if 'source' in accepted_vars and 'source' in user_params and user_params['source']:
        params['source'] = user_params['source']

    # tags, hosts, keys
    for var in ['tags', 'hosts', 'keys']:
        if var in accepted_vars and var in user_params and user_params[var]:
            params[var] = user_params[var].split(_list_separator)

    # fix or drop bad tags
    if params['tags']:
        params['tags'] = validate_tags(params['tags'])

    # operation for joining multiple tags
    if 'tags_op' in accepted_vars and 'tags_op' in user_params:
        op = user_params['tags_op']
        if op not in ['or', 'and']:
            # raise error or ignore
            if strict:
                raise ValueError(
                    f'Invalid operation: "{op}". Expected "or" or "and"')
        elif op == 'and':
            params['tags_op'] = 'and'

    # check keys
    if params['keys'] and strict:
        for key in params['keys']:
            if key not in _valid_entry_keys:
                raise ValueError(
                    f'Invalid key: {key}; expected {_valid_entry_keys}')

    # check if any of the filtering parameters have been set
    for var in ['since', 'until', 'tags', 'names', 'source', 'hosts']:
        if params[var]:
            params['_filtered'] = True
            break

    return params


def get_clean_params(params, accepted_vars, default=None):
    '''Clean parameters.'''
    if default is None:
        default = {}

    clean_params = {}
    for var in accepted_vars:
        # only pass along value if it isn't the default
        if ((var in params and params[var])
                and not (var in default and params[var] == default[var])):

            # cast int params
            if var in ['limit', 'offset']:
                params[var] = str(params[var])

            # join lists
            value = params[var]
            if isinstance(value, list):
                if var == 'names':
                    value = names_list_separator.join(value)
                else:
                    value = _list_separator.join(value)

            clean_params[var] = value

    return clean_params
