#! /usr/bin/env python3
'''Shared DB object'''
# pylint: disable = missing-docstring, global-statement

from os.path import dirname, join

from psqtie import DB as BaseDB


class DB(BaseDB):
    '''Loopvid DB'''
    _db_path = join(dirname(__file__), 'loopvid.db')


# will be created at the end of this file
db = None


def disable_auto_commit():
    '''Disable auto commit for the shared DB object.'''
    global db

    if db and not db.auto_commit:
        # nothing to do
        return

    db = DB(auto_commit=False)


def reenable_auto_commit():
    '''Re-enable auto commit for the shared DB object.'''
    global db

    if db and db.auto_commit:
        # nothing to do
        return

    # Make sure the connection is closed, so it will be properly garbage
    # collected. Note that it isn't a problem to call close() on an already
    # closed connection.
    if db:
        db.close()

    db = DB(auto_commit=True)


# expose psqtie.DB's methods as public functions, so we can replace the DB
# object as needed and immediately affect them all without any further action by
# modules importing db


def query(*args, **kwargs):
    return db.query(*args, **kwargs)


def fetch(*args, **kwargs):
    return db.fetch(*args, **kwargs)


def fetchall(*args, **kwargs):
    return db.fetchall(*args, **kwargs)


def fetchone(*args, **kwargs):
    return db.fetchone(*args, **kwargs)


def execute(*args, **kwargs):
    return db.execute(*args, **kwargs)


def select_from(*args, **kwargs):
    return db.select_from(*args, **kwargs)


def insert_into(*args, **kwargs):
    return db.insert_into(*args, **kwargs)


def update(*args, **kwargs):
    return db.update(*args, **kwargs)


def delete_from(*args, **kwargs):
    return db.delete_from(*args, **kwargs)


def commit():
    db.commit()


def rollback():
    db.rollback()


def close():
    db.close()


# static methods
row2dict = DB.row2dict


# create initial db object with auto_commit enabled
reenable_auto_commit()
