#! /usr/bin/env python3
'''Album pages'''

import re
from urllib.parse import unquote

from flask import request

from .common import Jinja, Shortener
from .db import split_names_list


class Album:
    '''Album pages'''
    shortened_list_regex = re.compile(r'^[a-z]{2}/[^&]+(&[a-z]{2}/[^&]+)*$')

    stylesheet_regex = re.compile(
        r'(?P<base>[^0-9]+)?(?P<number>[0-9]+)(?P<extension>\.css)?')

    # WARNING: remember to update if stylesheets are added or removed
    stylesheets = [
        'anonymous1',
        'anonymous2',
        'anonymous3',
    ]

    # WARNING: update if adding variables
    known_variables = [
        'n', 't', 'name', 'title', 'ss', 'sheet', 'stylesheet', 'u', 'unmute',
        'a', 'audio',
    ]

    @classmethod
    def get(cls, url_list=None):
        '''Takes a list of semi-colon separated shortened lists of urls, expands
        them and displays all videos in a single page.'''
        shortened_lists = []
        if url_list:
            url_list = unquote(url_list)
            shortened_lists = split_names_list(url_list)
        videos = []
        messages = []
        video_num = 0
        for shortened_list in shortened_lists:
            if not shortened_list:
                continue

            # each shortened_list corresponds to a list of mirrors for a
            # single video

            # strip superfluous slashes
            shortened_list = shortened_list.strip('/')
            # drop malformed entries
            if cls.shortened_list_regex.match(shortened_list):
                expanded_urls = Shortener.expand_urls(shortened_list)
                if expanded_urls:
                    # add file type to urls that contain an extension
                    urls_list = []
                    for url in expanded_urls:
                        urls_list.append({'url': url, 'type': Shortener.get_mime_type(url)})
                    video_num += 1
                    video = {
                        'number' : video_num,
                        'urls' : urls_list,
                        'watch_url' : '/'+shortened_list,
                    }
                    videos.append(video)
            else:
                messages.append('Unrecognized url: {}'.format(shortened_list))

        # variables
        title = cls.get_title()

        stylesheet, new_messages = cls.get_stylesheet()
        if new_messages:
            messages += new_messages

        audio, new_messages = cls.get_audio()
        if new_messages:
            messages += new_messages

        unmute, new_messages = cls.get_unmute(videos)
        if new_messages:
            messages += new_messages

        new_messages = cls.unknown_vars()
        if new_messages:
            messages += new_messages

        template_values = {
            'videos' : videos,
            'title' : title,
            'messages' : messages,
            'stylesheet' : stylesheet,
            'audio' : audio,
            'unmute' : unmute,
        }
        return Jinja.render('album', template_values)

    @staticmethod
    def get_title():
        '''Generate page title'''
        title = ''
        for variable in ['n', 't', 'name', 'title']:
            value = request.values.get(variable)
            if value:
                title = value
        return title

    @staticmethod
    def get_stylesheet():
        '''Choose stylesheet'''
        stylesheet = ''
        for variable in ['ss', 'sheet', 'stylesheet']:
            value = request.values.get(variable)
            if value:
                stylesheet = value
        if not stylesheet:
            return None, []

        match = Album.stylesheet_regex.match(stylesheet)
        if not match:
            return None, ['Unknown stylesheet: {}'.format(stylesheet)]

        # complete filename and drop extension
        if not match.group('base'):
            stylesheet = 'anonymous' + stylesheet
        # drop extension
        if match.group('extension'):
            stylesheet = stylesheet[:-4]

        # check that it exists
        if stylesheet in Album.stylesheets:
            return stylesheet, []
        return None, ['Unknown stylesheet: {}'.format(stylesheet)]

    @staticmethod
    def get_audio():
        '''Return audio track, if any'''
        messages = []
        audio = None
        for variable in ['a', 'audio']:
            value = request.values.get(variable)
            if value:
                audio = value
        if not audio:
            return None, messages

        expanded_audio = Shortener.expand_urls(audio, is_audio=True)
        audio_list = []
        if expanded_audio:
            for url in expanded_audio:
                audio_list.append({'url': url, 'type': Shortener.get_mime_type(url)})
        else:
            messages.append('Unknown url: {}'.format(audio))

        return audio_list, messages

    @staticmethod
    def get_unmute(videos):
        '''Return video to unmute, if any.'''
        messages = []
        unmute = None
        for variable in ['u', 'unmute']:
            value = request.values.get(variable)
            if value:

                try:
                    value = int(value)
                except ValueError:
                    messages.append('Invalid video number; can\'t unmute: {}'.format(value))
                    value = None

            # can't just test value here since 0 will test false
            if isinstance(value, int):
                if 1 <= value <= len(videos):
                    unmute = value
                else:
                    messages.append('No such video; can\'t unmute: {}'.format(value))

        if not unmute:
            return None, messages
        return unmute, messages

    @classmethod
    def unknown_vars(cls):
        '''Generate warnings if unknown variables are found.'''
        messages = []
        for variable in request.values:
            if not variable in cls.known_variables:
                messages.append('Unknown variable: {}'.format(variable))
        return messages
